<?php
require_once 'config.php';

if(!isLoggedIn()) {
    header("Location: login.php");
    exit();
}

if(!isActiveUser()) {
    header("Location: activate.php");
    exit();
}

$product_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if(!$product_id) {
    header("Location: products.php");
    exit();
}

// Get product details
$product_query = "
    SELECT p.*, c.name as category_name 
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    WHERE p.id = ? AND p.status = 'active'
";
$stmt = $conn->prepare($product_query);
$stmt->bind_param("i", $product_id);
$stmt->execute();
$product_result = $stmt->get_result();

if($product_result->num_rows == 0) {
    header("Location: products.php");
    exit();
}

$product = $product_result->fetch_assoc();

// Decode additional images
$additional_images = !empty($product['additional_images']) ? json_decode($product['additional_images'], true) : [];
$all_images = array_merge([$product['image_url']], $additional_images);

// Calculate discount
$discount = 0;
if($product['original_price'] > 0 && $product['sale_price'] > 0) {
    $discount = round((($product['original_price'] - $product['sale_price']) / $product['original_price']) * 100);
}

// Get related products
$related_products = $conn->query("
    SELECT * FROM products 
    WHERE category_id = {$product['category_id']} 
    AND id != {$product['id']} 
    AND status = 'active' 
    ORDER BY RAND() 
    LIMIT 4
");

// Process purchase
if($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['buy_now'])) {
    $telegram = sanitize($_POST['telegram_username']);
    $whatsapp = sanitize($_POST['whatsapp_number']);
    $customer_name = sanitize($_POST['customer_name']);
    
    if(empty($telegram) && empty($whatsapp)) {
        $error = "Please provide at least Telegram username or WhatsApp number";
    } else {
        $user_id = $_SESSION['user_id'];
        $user_balance = getUserBalance($user_id);
        
        if($user_balance >= $product['sale_price']) {
            // Start transaction
            $conn->begin_transaction();
            
            try {
                // Deduct balance
                $conn->query("UPDATE users SET balance = balance - {$product['sale_price']} WHERE id = $user_id");
                
                // Create order
                $order_stmt = $conn->prepare("
                    INSERT INTO orders (user_id, product_id, amount, telegram_username, whatsapp_number, customer_name, status) 
                    VALUES (?, ?, ?, ?, ?, ?, 'pending')
                ");
                $order_stmt->bind_param(
                    "iidsss", 
                    $user_id, 
                    $product_id, 
                    $product['sale_price'], 
                    $telegram, 
                    $whatsapp, 
                    $customer_name
                );
                $order_stmt->execute();
                
                // Record transaction
                $conn->query("
                    INSERT INTO transactions (user_id, type, amount, description, status) 
                    VALUES ($user_id, 'product_purchase', {$product['sale_price']}, 'Purchase: {$product['name']}', 'completed')
                ");
                
                $conn->commit();
                
                $success = "Purchase successful! Your order has been placed.";
                header("refresh:2;url=my-orders.php");
            } catch (Exception $e) {
                $conn->rollback();
                $error = "Purchase failed. Please try again.";
            }
        } else {
            $error = "Insufficient balance. Please deposit first.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($product['name']); ?> - Project Sale</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        .image-gallery img {
            transition: transform 0.3s ease;
        }
        .image-gallery img:hover {
            transform: scale(1.05);
        }
        .sticky-sidebar {
            position: sticky;
            top: 100px;
        }
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
            animation: fadeIn 0.5s ease;
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Include Header -->
    <?php include 'includes/header.php'; ?>
    
    <div class="flex min-h-screen">
        <!-- Sidebar -->
        <div class="hidden md:block w-64 bg-white shadow-lg">
            <?php include 'includes/sidebar.php'; ?>
        </div>
        
        <!-- Main Content -->
        <div class="flex-1 p-6">
            <!-- Breadcrumb -->
            <div class="mb-6">
                <nav class="flex" aria-label="Breadcrumb">
                    <ol class="inline-flex items-center space-x-1 md:space-x-3">
                        <li class="inline-flex items-center">
                            <a href="index.php" class="inline-flex items-center text-sm text-gray-700 hover:text-blue-600">
                                <i class="fas fa-home mr-2"></i> Home
                            </a>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <a href="products.php" class="ml-1 text-sm text-gray-700 hover:text-blue-600 md:ml-2">Products</a>
                            </div>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <a href="products.php?category=<?php echo $product['category_id']; ?>" 
                                   class="ml-1 text-sm text-gray-700 hover:text-blue-600 md:ml-2">
                                    <?php echo $product['category_name']; ?>
                                </a>
                            </div>
                        </li>
                        <li aria-current="page">
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <span class="ml-1 text-sm text-gray-500 md:ml-2 font-medium">
                                    <?php echo substr($product['name'], 0, 30); ?>...
                                </span>
                            </div>
                        </li>
                    </ol>
                </nav>
            </div>
            
            <!-- Messages -->
            <?php if($error): ?>
                <div class="bg-red-50 border-l-4 border-red-500 p-4 mb-6 rounded-lg animate__animated animate__shakeX">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-3 text-xl"></i>
                        <span class="text-red-700 font-medium"><?php echo $error; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if($success): ?>
                <div class="bg-green-50 border-l-4 border-green-500 p-4 mb-6 rounded-lg animate__animated animate__fadeIn">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-3 text-xl"></i>
                        <span class="text-green-700 font-medium"><?php echo $success; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Product Details -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8 mb-12">
                <!-- Left Column - Images -->
                <div class="lg:col-span-2 space-y-6">
                    <!-- Main Image -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden">
                        <div class="p-4 border-b flex justify-between items-center">
                            <h3 class="text-lg font-bold text-gray-800">
                                <i class="fas fa-images mr-2"></i> Product Images
                            </h3>
                            <span class="text-gray-500 text-sm">
                                <?php echo count($all_images); ?> images
                            </span>
                        </div>
                        <div class="p-6">
                            <div class="relative h-96 rounded-lg overflow-hidden mb-6">
                                <img id="mainImage" 
                                     src="<?php echo $all_images[0]; ?>" 
                                     alt="<?php echo htmlspecialchars($product['name']); ?>"
                                     class="w-full h-full object-cover">
                                
                                <!-- Discount Badge -->
                                <?php if($discount > 0): ?>
                                <div class="absolute top-4 right-4 bg-gradient-to-r from-red-500 to-pink-600 text-white px-4 py-2 rounded-full font-bold shadow-lg">
                                    -<?php echo $discount; ?>% OFF
                                </div>
                                <?php endif; ?>
                            </div>
                            
                            <!-- Thumbnail Gallery -->
                            <?php if(count($all_images) > 1): ?>
                            <div class="image-gallery grid grid-cols-4 md:grid-cols-6 gap-3">
                                <?php foreach($all_images as $index => $image): ?>
                                <button onclick="changeMainImage('<?php echo $image; ?>')" 
                                        class="border-2 border-transparent hover:border-blue-500 rounded-lg overflow-hidden transition duration-300">
                                    <img src="<?php echo $image; ?>" 
                                         alt="Thumbnail <?php echo $index + 1; ?>"
                                         class="w-full h-20 object-cover">
                                </button>
                                <?php endforeach; ?>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Product Description Tabs -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden">
                        <!-- Tabs Header -->
                        <div class="border-b">
                            <div class="flex overflow-x-auto">
                                <button onclick="showTab('description')" 
                                        class="tab-btn flex-1 px-6 py-4 text-center font-medium border-b-2 border-transparent hover:text-blue-600 hover:border-blue-300 transition duration-300">
                                    <i class="fas fa-file-alt mr-2"></i> Description
                                </button>
                                <button onclick="showTab('specifications')" 
                                        class="tab-btn flex-1 px-6 py-4 text-center font-medium border-b-2 border-transparent hover:text-blue-600 hover:border-blue-300 transition duration-300">
                                    <i class="fas fa-list mr-2"></i> Specifications
                                </button>
                                <button onclick="showTab('delivery')" 
                                        class="tab-btn flex-1 px-6 py-4 text-center font-medium border-b-2 border-transparent hover:text-blue-600 hover:border-blue-300 transition duration-300">
                                    <i class="fas fa-shipping-fast mr-2"></i> Delivery
                                </button>
                                <button onclick="showTab('reviews')" 
                                        class="tab-btn flex-1 px-6 py-4 text-center font-medium border-b-2 border-transparent hover:text-blue-600 hover:border-blue-300 transition duration-300">
                                    <i class="fas fa-star mr-2"></i> Reviews
                                </button>
                            </div>
                        </div>
                        
                        <!-- Tabs Content -->
                        <div class="p-6">
                            <!-- Description Tab -->
                            <div id="description" class="tab-content active">
                                <div class="prose max-w-none">
                                    <?php echo nl2br(htmlspecialchars($product['description'])); ?>
                                </div>
                                
                                <!-- Features -->
                                <div class="mt-8 grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div class="flex items-center p-4 bg-blue-50 rounded-lg">
                                        <i class="fas fa-check-circle text-green-500 text-xl mr-4"></i>
                                        <div>
                                            <h4 class="font-bold text-gray-800">Instant Delivery</h4>
                                            <p class="text-gray-600 text-sm">Get product immediately after purchase</p>
                                        </div>
                                    </div>
                                    <div class="flex items-center p-4 bg-green-50 rounded-lg">
                                        <i class="fas fa-check-circle text-green-500 text-xl mr-4"></i>
                                        <div>
                                            <h4 class="font-bold text-gray-800">Quality Guaranteed</h4>
                                            <p class="text-gray-600 text-sm">Premium quality digital products</p>
                                        </div>
                                    </div>
                                    <div class="flex items-center p-4 bg-purple-50 rounded-lg">
                                        <i class="fas fa-check-circle text-green-500 text-xl mr-4"></i>
                                        <div>
                                            <h4 class="font-bold text-gray-800">24/7 Support</h4>
                                            <p class="text-gray-600 text-sm">Round the clock customer support</p>
                                        </div>
                                    </div>
                                    <div class="flex items-center p-4 bg-yellow-50 rounded-lg">
                                        <i class="fas fa-check-circle text-green-500 text-xl mr-4"></i>
                                        <div>
                                            <h4 class="font-bold text-gray-800">Secure Payment</h4>
                                            <p class="text-gray-600 text-sm">100% secure transaction</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Specifications Tab -->
                            <div id="specifications" class="tab-content">
                                <div class="space-y-4">
                                    <div class="flex justify-between py-3 border-b">
                                        <span class="font-medium text-gray-700">Product Name</span>
                                        <span class="text-gray-800"><?php echo $product['name']; ?></span>
                                    </div>
                                    <div class="flex justify-between py-3 border-b">
                                        <span class="font-medium text-gray-700">Category</span>
                                        <span class="text-gray-800"><?php echo $product['category_name']; ?></span>
                                    </div>
                                    <div class="flex justify-between py-3 border-b">
                                        <span class="font-medium text-gray-700">Original Price</span>
                                        <span class="text-gray-800">
                                            <?php if($product['original_price'] > 0): ?>
                                                ৳<?php echo number_format($product['original_price'], 2); ?>
                                            <?php else: ?>
                                                N/A
                                            <?php endif; ?>
                                        </span>
                                    </div>
                                    <div class="flex justify-between py-3 border-b">
                                        <span class="font-medium text-gray-700">Sale Price</span>
                                        <span class="text-2xl font-bold text-blue-600">
                                            ৳<?php echo number_format($product['sale_price'], 2); ?>
                                        </span>
                                    </div>
                                    <div class="flex justify-between py-3 border-b">
                                        <span class="font-medium text-gray-700">Discount</span>
                                        <span class="font-bold <?php echo $discount > 0 ? 'text-green-600' : 'text-gray-600'; ?>">
                                            <?php echo $discount > 0 ? $discount . '% OFF' : 'No discount'; ?>
                                        </span>
                                    </div>
                                    <div class="flex justify-between py-3 border-b">
                                        <span class="font-medium text-gray-700">Added On</span>
                                        <span class="text-gray-800">
                                            <?php echo date('F d, Y', strtotime($product['created_at'])); ?>
                                        </span>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Delivery Tab -->
                            <div id="delivery" class="tab-content">
                                <div class="space-y-6">
                                    <div class="p-4 bg-blue-50 rounded-lg">
                                        <h4 class="font-bold text-gray-800 mb-2">
                                            <i class="fas fa-info-circle mr-2"></i> Delivery Information
                                        </h4>
                                        <p class="text-gray-600">
                                            This is a digital product. After successful purchase, you will receive:
                                        </p>
                                        <ul class="list-disc list-inside text-gray-600 mt-2 space-y-1">
                                            <li>Instant access to download links</li>
                                            <li>Product files via Telegram/WhatsApp</li>
                                            <li>Complete documentation and support</li>
                                            <li>24/7 customer assistance</li>
                                        </ul>
                                    </div>
                                    
                                    <div class="p-4 bg-green-50 rounded-lg">
                                        <h4 class="font-bold text-gray-800 mb-2">
                                            <i class="fas fa-clock mr-2"></i> Delivery Time
                                        </h4>
                                        <p class="text-gray-600">
                                            Delivery is instant! You'll receive the product within minutes of successful payment.
                                        </p>
                                    </div>
                                    
                                    <div class="p-4 bg-purple-50 rounded-lg">
                                        <h4 class="font-bold text-gray-800 mb-2">
                                            <i class="fas fa-headset mr-2"></i> Support
                                        </h4>
                                        <p class="text-gray-600">
                                            Need help? Contact our support team via:
                                        </p>
                                        <div class="flex flex-wrap gap-3 mt-3">
                                            <a href="https://t.me/<?php echo $product['telegram_support'] ?? 'support'; ?>" 
                                               target="_blank" 
                                               class="inline-flex items-center bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 transition duration-300">
                                                <i class="fab fa-telegram mr-2"></i> Telegram Support
                                            </a>
                                            <a href="https://wa.me/<?php echo $product['whatsapp_support'] ?? '8801234567890'; ?>" 
                                               target="_blank" 
                                               class="inline-flex items-center bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 transition duration-300">
                                                <i class="fab fa-whatsapp mr-2"></i> WhatsApp Support
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Reviews Tab -->
                            <div id="reviews" class="tab-content">
                                <div class="text-center py-8">
                                    <div class="w-20 h-20 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                        <i class="fas fa-comments text-gray-400 text-3xl"></i>
                                    </div>
                                    <h4 class="text-xl font-bold text-gray-800 mb-2">No Reviews Yet</h4>
                                    <p class="text-gray-600 mb-6">Be the first to review this product!</p>
                                    <button onclick="showPurchaseForm()" 
                                            class="bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300">
                                        <i class="fas fa-shopping-cart mr-2"></i> Purchase & Review
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Right Column - Purchase Section -->
                <div class="space-y-6">
                    <!-- Product Info Card -->
                    <div class="sticky-sidebar bg-white rounded-xl shadow-lg p-6">
                        <!-- Product Title -->
                        <h1 class="text-2xl font-bold text-gray-800 mb-4">
                            <?php echo htmlspecialchars($product['name']); ?>
                        </h1>
                        
                        <!-- Category -->
                        <div class="mb-4">
                            <span class="inline-flex items-center px-3 py-1 bg-blue-100 text-blue-700 rounded-full text-sm font-medium">
                                <i class="fas fa-tag mr-1"></i>
                                <?php echo $product['category_name']; ?>
                            </span>
                        </div>
                        
                        <!-- Price -->
                        <div class="mb-6">
                            <?php if($product['original_price'] > 0): ?>
                                <div class="flex items-center mb-2">
                                    <span class="text-gray-400 line-through text-lg">
                                        ৳<?php echo number_format($product['original_price'], 2); ?>
                                    </span>
                                    <span class="ml-3 bg-red-100 text-red-700 px-2 py-1 rounded text-sm font-bold">
                                        Save ৳<?php echo number_format($product['original_price'] - $product['sale_price'], 2); ?>
                                    </span>
                                </div>
                            <?php endif; ?>
                            <div class="text-4xl font-bold text-gray-800">
                                ৳<?php echo number_format($product['sale_price'], 2); ?>
                            </div>
                            <?php if($discount > 0): ?>
                                <div class="text-green-600 font-medium mt-1">
                                    <i class="fas fa-bolt mr-1"></i> <?php echo $discount; ?>% Discount Applied
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- User Balance -->
                        <div class="mb-6 p-4 bg-gray-50 rounded-lg">
                            <div class="flex justify-between items-center mb-2">
                                <span class="text-gray-600">Your Balance:</span>
                                <span class="text-xl font-bold <?php echo getUserBalance($_SESSION['user_id']) >= $product['sale_price'] ? 'text-green-600' : 'text-red-600'; ?>">
                                    ৳<?php echo number_format(getUserBalance($_SESSION['user_id']), 2); ?>
                                </span>
                            </div>
                            <?php if(getUserBalance($_SESSION['user_id']) < $product['sale_price']): ?>
                                <div class="text-red-600 text-sm mt-2">
                                    <i class="fas fa-exclamation-triangle mr-1"></i>
                                    Need ৳<?php echo number_format($product['sale_price'] - getUserBalance($_SESSION['user_id']), 2); ?> more
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Purchase Form -->
                        <form method="POST" action="" class="space-y-4" id="purchaseForm">
                            <div>
                                <label class="block text-gray-700 text-sm font-bold mb-2">
                                    <i class="fas fa-user mr-1"></i> Your Name
                                </label>
                                <input type="text" name="customer_name" required
                                       class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                       placeholder="Enter your full name"
                                       value="<?php echo $_SESSION['username']; ?>">
                            </div>
                            
                            <div>
                                <label class="block text-gray-700 text-sm font-bold mb-2">
                                    <i class="fab fa-telegram text-blue-500 mr-1"></i> Telegram Username
                                </label>
                                <input type="text" name="telegram_username"
                                       class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                       placeholder="@username (optional)">
                            </div>
                            
                            <div>
                                <label class="block text-gray-700 text-sm font-bold mb-2">
                                    <i class="fab fa-whatsapp text-green-500 mr-1"></i> WhatsApp Number
                                </label>
                                <input type="text" name="whatsapp_number"
                                       class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                       placeholder="+8801XXXXXXXXX (optional)">
                            </div>
                            
                            <div class="pt-4">
                                <button type="submit" name="buy_now"
                                        class="w-full bg-gradient-to-r from-green-500 to-emerald-600 text-white font-bold py-4 px-6 rounded-lg text-lg hover:from-green-600 hover:to-emerald-700 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 transition duration-300 transform hover:scale-105 disabled:opacity-50 disabled:cursor-not-allowed"
                                        <?php echo getUserBalance($_SESSION['user_id']) < $product['sale_price'] ? 'disabled' : ''; ?>>
                                    <i class="fas fa-shopping-cart mr-2"></i> 
                                    <?php echo getUserBalance($_SESSION['user_id']) >= $product['sale_price'] ? 'Buy Now' : 'Insufficient Balance'; ?>
                                </button>
                                
                                <?php if(getUserBalance($_SESSION['user_id']) < $product['sale_price']): ?>
                                    <div class="mt-3">
                                        <a href="deposit.php" 
                                           class="block w-full text-center bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300">
                                            <i class="fas fa-plus-circle mr-2"></i> Deposit Now
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </form>
                        
                        <!-- Quick Actions -->
                        <div class="mt-6 pt-6 border-t">
                            <div class="grid grid-cols-2 gap-3">
                                <button onclick="addToCart(<?php echo $product['id']; ?>)" 
                                        class="bg-gradient-to-r from-blue-500 to-indigo-600 text-white font-bold py-3 px-4 rounded-lg hover:from-blue-600 hover:to-indigo-700 transition duration-300">
                                    <i class="fas fa-cart-plus mr-2"></i> Add to Cart
                                </button>
                                <button onclick="shareProduct()" 
                                        class="bg-gradient-to-r from-purple-500 to-pink-600 text-white font-bold py-3 px-4 rounded-lg hover:from-purple-600 hover:to-pink-700 transition duration-300">
                                    <i class="fas fa-share-alt mr-2"></i> Share
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Related Products -->
                    <?php if($related_products->num_rows > 0): ?>
                    <div class="bg-white rounded-xl shadow-lg p-6">
                        <h3 class="text-xl font-bold text-gray-800 mb-6">
                            <i class="fas fa-random mr-2"></i> Related Products
                        </h3>
                        <div class="space-y-4">
                            <?php while($related = $related_products->fetch_assoc()): ?>
                            <a href="product-details.php?id=<?php echo $related['id']; ?>" 
                               class="flex items-center p-3 border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition duration-300 group">
                                <div class="w-16 h-16 rounded-lg overflow-hidden mr-4">
                                    <img src="<?php echo $related['image_url'] ?: 'https://via.placeholder.com/100'; ?>" 
                                         alt="<?php echo htmlspecialchars($related['name']); ?>"
                                         class="w-full h-full object-cover group-hover:scale-110 transition duration-300">
                                </div>
                                <div class="flex-1">
                                    <h4 class="font-medium text-gray-800 group-hover:text-blue-600">
                                        <?php echo substr($related['name'], 0, 30); ?>...
                                    </h4>
                                    <p class="text-lg font-bold text-gray-800">
                                        ৳<?php echo number_format($related['sale_price'], 2); ?>
                                    </p>
                                </div>
                                <i class="fas fa-chevron-right text-gray-300 group-hover:text-blue-500"></i>
                            </a>
                            <?php endwhile; ?>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Support Card -->
                    <div class="bg-gradient-to-r from-blue-500 to-indigo-600 rounded-xl shadow-lg p-6 text-white">
                        <h3 class="text-xl font-bold mb-4">
                            <i class="fas fa-headset mr-2"></i> Need Help?
                        </h3>
                        <p class="mb-6 opacity-90">
                            Our support team is available 24/7 to assist you with any questions.
                        </p>
                        <div class="space-y-3">
                            <a href="https://t.me/support" target="_blank" 
                               class="flex items-center justify-center bg-white text-blue-600 font-bold py-3 px-4 rounded-lg hover:bg-blue-50 transition duration-300">
                                <i class="fab fa-telegram mr-2 text-xl"></i> Telegram Support
                            </a>
                            <a href="https://wa.me/8801234567890" target="_blank" 
                               class="flex items-center justify-center bg-white text-green-600 font-bold py-3 px-4 rounded-lg hover:bg-green-50 transition duration-300">
                                <i class="fab fa-whatsapp mr-2 text-xl"></i> WhatsApp Support
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Share Modal -->
    <div id="shareModal" class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden items-center justify-center p-4">
        <div class="bg-white rounded-2xl max-w-md w-full">
            <div class="p-6">
                <h3 class="text-xl font-bold text-gray-800 mb-4">
                    <i class="fas fa-share-alt mr-2"></i> Share Product
                </h3>
                <div class="space-y-4">
                    <div>
                        <label class="block text-gray-700 text-sm font-bold mb-2">Share Link</label>
                        <div class="flex">
                            <input type="text" id="shareLink" 
                                   value="<?php echo SITE_URL . '/product-details.php?id=' . $product['id']; ?>" 
                                   readonly class="flex-1 px-4 py-2 border border-gray-300 rounded-l-lg bg-gray-50">
                            <button onclick="copyShareLink()" 
                                    class="bg-blue-600 text-white px-4 py-2 rounded-r-lg hover:bg-blue-700 transition duration-300">
                                <i class="fas fa-copy"></i>
                            </button>
                        </div>
                    </div>
                    
                    <div>
                        <label class="block text-gray-700 text-sm font-bold mb-2">Share with Referral</label>
                        <div class="flex">
                            <input type="text" id="shareLinkWithRef" 
                                   value="<?php echo SITE_URL . '/register.php?referral=' . ($_SESSION['referral_code'] ?? ''); ?>&product=<?php echo $product['id']; ?>" 
                                   readonly class="flex-1 px-4 py-2 border border-gray-300 rounded-l-lg bg-gray-50">
                            <button onclick="copyShareLinkWithRef()" 
                                    class="bg-purple-600 text-white px-4 py-2 rounded-r-lg hover:bg-purple-700 transition duration-300">
                                <i class="fas fa-copy"></i>
                            </button>
                        </div>
                        <p class="text-xs text-gray-500 mt-2">Earn commission when someone buys through this link</p>
                    </div>
                    
                    <!-- Social Share -->
                    <div class="pt-4">
                        <h4 class="font-medium text-gray-700 mb-3">Share on Social Media</h4>
                        <div class="flex justify-center space-x-4">
                            <a href="https://facebook.com/sharer/sharer.php?u=<?php echo urlencode(SITE_URL . '/product-details.php?id=' . $product['id']); ?>" 
                               target="_blank" 
                               class="w-12 h-12 bg-blue-600 text-white rounded-full flex items-center justify-center hover:bg-blue-700 transition duration-300">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                            <a href="https://twitter.com/intent/tweet?text=<?php echo urlencode('Check out this product: ' . $product['name']); ?>&url=<?php echo urlencode(SITE_URL . '/product-details.php?id=' . $product['id']); ?>" 
                               target="_blank" 
                               class="w-12 h-12 bg-sky-500 text-white rounded-full flex items-center justify-center hover:bg-sky-600 transition duration-300">
                                <i class="fab fa-twitter"></i>
                            </a>
                            <a href="https://wa.me/?text=<?php echo urlencode('Check out this product: ' . $product['name'] . ' ' . SITE_URL . '/product-details.php?id=' . $product['id']); ?>" 
                               target="_blank" 
                               class="w-12 h-12 bg-green-500 text-white rounded-full flex items-center justify-center hover:bg-green-600 transition duration-300">
                                <i class="fab fa-whatsapp"></i>
                            </a>
                            <a href="https://t.me/share/url?url=<?php echo urlencode(SITE_URL . '/product-details.php?id=' . $product['id']); ?>&text=<?php echo urlencode('Check out this product: ' . $product['name']); ?>" 
                               target="_blank" 
                               class="w-12 h-12 bg-blue-400 text-white rounded-full flex items-center justify-center hover:bg-blue-500 transition duration-300">
                                <i class="fab fa-telegram"></i>
                            </a>
                        </div>
                    </div>
                </div>
                
                <div class="mt-8 flex justify-end">
                    <button onclick="closeShareModal()" 
                            class="bg-gray-200 text-gray-700 font-bold py-2 px-6 rounded-lg hover:bg-gray-300 transition duration-300">
                        Close
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    // Image Gallery
    function changeMainImage(src) {
        document.getElementById('mainImage').src = src;
    }
    
    // Tab System
    function showTab(tabId) {
        // Hide all tabs
        document.querySelectorAll('.tab-content').forEach(tab => {
            tab.classList.remove('active');
        });
        
        // Remove active class from all buttons
        document.querySelectorAll('.tab-btn').forEach(btn => {
            btn.classList.remove('border-blue-500', 'text-blue-600');
            btn.classList.add('border-transparent');
        });
        
        // Show selected tab
        document.getElementById(tabId).classList.add('active');
        
        // Add active class to clicked button
        event.target.classList.remove('border-transparent');
        event.target.classList.add('border-blue-500', 'text-blue-600');
    }
    
    // Share Product
    function shareProduct() {
        document.getElementById('shareModal').classList.remove('hidden');
        document.getElementById('shareModal').classList.add('flex');
    }
    
    function closeShareModal() {
        document.getElementById('shareModal').classList.add('hidden');
        document.getElementById('shareModal').classList.remove('flex');
    }
    
    function copyShareLink() {
        const shareLink = document.getElementById('shareLink');
        shareLink.select();
        document.execCommand('copy');
        showNotification('Link copied to clipboard!', 'success');
    }
    
    function copyShareLinkWithRef() {
        const shareLink = document.getElementById('shareLinkWithRef');
        shareLink.select();
        document.execCommand('copy');
        showNotification('Referral link copied!', 'success');
    }
    
    // Add to Cart
    function addToCart(productId) {
        fetch('ajax/add-to-cart.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: 'product_id=' + productId
        })
        .then(response => response.json())
        .then(data => {
            if(data.success) {
                showNotification('Product added to cart!', 'success');
            } else {
                showNotification(data.message || 'Error adding to cart', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Network error', 'error');
        });
    }
    
    function showPurchaseForm() {
        document.getElementById('purchaseForm').scrollIntoView({ 
            behavior: 'smooth' 
        });
        document.getElementById('purchaseForm').querySelector('input').focus();
    }
    
    function showNotification(message, type) {
        // Remove existing notification
        const existing = document.querySelector('.notification-toast');
        if(existing) existing.remove();
        
        // Create notification
        const notification = document.createElement('div');
        notification.className = `notification-toast fixed top-4 right-4 px-6 py-3 rounded-lg shadow-xl text-white font-bold z-50 animate__animated animate__fadeInRight ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;
        notification.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-3 text-xl"></i>
                <span>${message}</span>
            </div>
        `;
        document.body.appendChild(notification);
        
        // Remove after 3 seconds
        setTimeout(() => {
            notification.classList.remove('animate__fadeInRight');
            notification.classList.add('animate__fadeOutRight');
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }
    
    // Close modals when clicking outside
    document.addEventListener('click', function(event) {
        const shareModal = document.getElementById('shareModal');
        const shareContent = shareModal.querySelector('.bg-white');
        
        if(shareModal.classList.contains('flex') && !shareContent.contains(event.target) && !event.target.closest('[onclick*="shareProduct"]')) {
            closeShareModal();
        }
    });
    </script>
</body>
</html>